#!/usr/bin/perl -w
use strict;

use Plack::Test;
use HTTP::Request::Common;

use Test::More;
#use Test::WWW::Mechanize::CGIApp;
use Test::JSON;
use Digest::MD5 qw{md5_hex};

# test fixture setup
my $dbname = 't/apptest.db';
unlink $dbname if -e $dbname;
my $bookmarks = Bookmarks->new({ dbname => $dbname });
$bookmarks->create_tables;
my $username = 'apptest';
my $password = 'apptest';

# app setup
use BookmarksApp;
my $app = BookmarksApp->new({
    config => {
        dbname          => $dbname,
        #auth            => 1,
        digest_key      => 'test_secret',
        digest_password => md5_hex("$username:Bookmarks:$password"),
        #proxy_ip        => '',
    }
})->to_app;

my $test = Plack::Test->create($app);

my $res = $test->request(GET "/");
is($res->content_type, 'text/html');

$res = $test->request(GET '/?format=json');
is($res->content_type, 'application/json', 'JSON content type is application/json');
is_valid_json $res->content;
is_json $res->content, '{ "bookmarks": [] }';

$res = $test->request(GET '/?format=xbel');
is($res->content_type, 'application/xml', 'XBEL content type is application/xml');

$res = $test->request(GET '/?format=text');
is($res->content_type, 'text/uri-list', 'Text content type is text/uri-list');


my $bookmark_uri;

subtest 'Created bookmark' => sub {
    # create a bookmark
    my $res = $test->request(POST
        '/',
        {
            uri => 'http://metacpan.org',
            title => 'My Bookmark',
            tags => 'test',
        },
    );

    $bookmark_uri = $res->header('Location'); #->as_string;

    $res = $test->request(GET "$bookmark_uri/title");
    is($res->content_type, 'text/plain', 'Title is text/plain');
    is($res->content, 'My Bookmark', 'Title has correct value');
    $res = $test->request(GET "$bookmark_uri/uri");
    is($res->content_type, 'text/plain', 'URI is text/plain');
    is($res->content, 'http://metacpan.org', 'URI has correct value');
    $res = $test->request(GET "$bookmark_uri/tags");
    is($res->content_type, 'text/plain', 'Tags are text/plain');
    is($res->content, 'test', 'Tags have correct value');
};

subtest 'Updated bookmark' => sub {
    my $res = $test->request(
        POST $bookmark_uri,
        {
            uri => 'http://search.cpan.org',
            title => 'My Other Bookmark',
            tags => 'test one two',
        },
    );
    $res = $test->request(GET "$bookmark_uri/title");
    is($res->content_type, 'text/plain', 'Title is text/plain');
    is($res->content, 'My Other Bookmark', 'Title has correct value');
    $res = $test->request(GET "$bookmark_uri/uri");
    is($res->content_type, 'text/plain', 'URI is text/plain');
    is($res->content, 'http://search.cpan.org', 'URI has correct value');
    $res = $test->request(GET "$bookmark_uri/tags");
    is($res->content_type, 'text/plain', 'Tags are text/plain');
    is($res->content, 'one test two', 'Tags have correct value');
};

$res = $test->request(GET "$bookmark_uri/id");
my $id = $res->content;
$res = $test->request(GET "$bookmark_uri/ctime");
my $ctime = $res->content;
$res = $test->request(GET "$bookmark_uri/mtime");
my $mtime = $res->content;

$res = $test->request(GET "$bookmark_uri?format=json");
is($res->content_type, 'application/json');
is_valid_json $res->content;
is_json $res->content, <<"END_JSON";
{
    "id": $id,
    "bookmark_uri": "$bookmark_uri",
    "ctime": $ctime,
    "mtime": $mtime,
    "uri": "http://search.cpan.org",
    "title": "My Other Bookmark",
    "tags": ["one", "test", "two"]
}
END_JSON

done_testing();

# test fixture teardown
unlink $dbname if -e $dbname;
