#!/usr/bin/perl -w
use strict;

use YAML;
use Plack::Builder;
use Plack::Request;
use Router::Resource;

use Bookmarks::Controller;

#TODO: allow individual options to be set via environment vars, too

-e $ENV{CONFIG_FILE} or die "Config file $ENV{CONFIG_FILE} not found\n";

my $config = YAML::LoadFile($ENV{CONFIG_FILE});

sub get_controller {
    my $env = shift;
    my $req = Plack::Request->new($env);

    return Bookmarks::Controller->new({
        request => $req,
        dbname  => $config->{dbname},
    });
}

my $router = router {
    resource '/' => sub {
        GET {
            my ($env) = @_;
            my $controller = get_controller($env);

            # check for a uri param, and if there is one present,
            # see if a bookmark for that URI already exists; if so
            # redirect to that bookmark, and if not, show the form
            # to create a new bookmark
            if (defined $controller->request->param('uri')) {
                return $controller->find_or_new;
            }

            # otherwise return a list
            return $controller->list;
        };
        POST {
            my ($env) = @_;
            my $controller = get_controller($env);

            # create the bookmark and redirect to the new bookmark's edit form
            return $controller->create_and_redirect;
        };
    };

    resource '/list' => sub {
        GET {
            my ($env) = @_;
            my $controller = get_controller($env);

            return $controller->list;
        };
    };

    resource '/feed' => sub {
        GET {
            my ($env) = @_;
            my $controller = get_controller($env);

            return $controller->feed;
        };
    };

    resource '/{id}' => sub {
        GET {
            my ($env, $params) = @_;
            my $controller = get_controller($env);

            return $controller->view($params->{id});
        };
        POST {
            my ($env, $params) = @_;
            my $controller = get_controller($env);

            return $controller->update_and_redirect($params->{id});
        };
    };

    resource '/{id}/{field}' => sub {
        GET {
            my ($env, $params) = @_;
            my $controller = get_controller($env);

            return $controller->view_field($params->{id}, $params->{field});
        };
    };
};

builder {
    enable_if { $_[0]->{REMOTE_ADDR} eq $config->{proxy_ip} } 'ReverseProxy';
    enable_if { $_[0]->{REQUEST_METHOD} ne 'GET' } 'Auth::Digest', (
        realm           => 'Bookmarks',
        secret          => $config->{digest_key},
        password_hashed => 1,
        authenticator   => sub { $config->{digest_password} }
    );
    sub { $router->dispatch(shift); };
};
