#!/usr/bin/perl -w
use strict;

use YAML;
use Plack::Builder;
use Plack::Request;
use Router::Resource;

use BookmarkController;
use Bookmarks;

#TODO: allow a different config file on the command line, or set options from the command line

-e 'conf.yml' or die "Missing required conf.yml config file\n";

my $config = YAML::LoadFile('conf.yml');

sub get_controller {
    my $req = shift;

    my $controller = BookmarkController->new({
        request => $req,
    });
    my $bookmarks = Bookmarks->new({
        dbname   => $config->{dbname},
        base_uri => $controller->base_uri,
    });
    $controller->bookmarks($bookmarks);
    return $controller;
}

my $router = router {
    resource '/' => sub {
        GET {
            my ($env) = @_;
            my $req = Plack::Request->new($env);
            my $controller = get_controller($req);

            # check for a uri param, and if there is one present,
            # see if a bookmark for that URI already exists; if so
            # redirect to that bookmark, and if not, show the form
            # to create a new bookmark
            if (defined $req->param('uri')) {
                return $controller->find_or_new;
            }

            # otherwise return a list
            return $controller->list;
        };
        POST {
            my ($env) = @_;
            my $req = Plack::Request->new($env);
            my $controller = get_controller($req);

            # create the bookmark and redirect to the new bookmark's edit form
            return $controller->create;
        };
    };

    resource '/list' => sub {
        GET {
            my ($env) = @_;
            my $req = Plack::Request->new($env);
            my $controller = get_controller($req);

            return $controller->list;
        };
    };

    resource '/feed' => sub {
        GET {
            my ($env) = @_;
            my $req = Plack::Request->new($env);
            my $controller = get_controller($req);

            return $controller->feed;
        };
    };

    resource '/{id}' => sub {
        GET {
            my ($env, $params) = @_;
            my $req = Plack::Request->new($env);
            my $controller = get_controller($req);

            return $controller->view($params->{id});
        };
        POST {
            my ($env, $params) = @_;
            my $req = Plack::Request->new($env);
            my $controller = get_controller($req);

            return $controller->edit($params->{id});

            return [200, ['Content-Type' => 'text/plain'], ['update ', $params->{id}]];
        };
    };

    resource '/{id}/{field}' => sub {
        GET {
            my ($env, $params) = @_;
            my $req = Plack::Request->new($env);
            my $controller = get_controller($req);

            return $controller->view_field($params->{id}, $params->{field});
        };
    };
};

#builder {
#    sub { $router->dispatch(shift); };
#};

builder {
    enable_if { $_[0]->{REMOTE_ADDR} eq $config->{proxy_ip} } 'ReverseProxy';
    enable_if { $_[0]->{REQUEST_METHOD} ne 'GET' } 'Auth::Digest', (
        realm           => 'Bookmarks',
        secret          => $config->{digest_key},
        password_hashed => 1,
        authenticator   => sub { $config->{digest_password} }
    );
    sub { $router->dispatch(shift); };
};
