package BookmarkApp;
use strict;
use warnings;
use base qw{CGI::Application};

use DBI;
use Encode;
use JSON;
use Template;
use Bookmarks;

my $template = Template->new;

my $dbname = 'new.db';
my $dbh = DBI->connect("dbi:SQLite:dbname=$dbname", "", "", { RaiseError => 1 });
my $bookmarks = Bookmarks->new({
    dbh => $dbh,
});

sub setup {
    my $self = shift;
    $self->mode_param(path_info => 1);
    $self->run_modes([qw{
        list
        view
        edit
    }]);
}

sub list {
    my $self = shift;
    my $q = $self->query;

    # check for a uri param, and if there is one present,
    # see if a bookmark for that URI already exists
    if (defined(my $uri = $q->param('uri'))) {
        my $bookmark = $bookmarks->get_bookmark({ uri => $uri });
        if ($bookmark) {
            # redirect to the view of the existing bookmark
            $self->header_type('redirect');
            $self->header_props(
                -uri => $q->url . '/' . $bookmark->{id},
            );
            return;
        } else {
            # bookmark was not found; show the form to create a new bookmark
            $bookmark->{uri} = $uri;
            $bookmark->{title} = $q->param('title');
            $self->header_props(
                -type    => 'text/html',
                -charset => 'UTF-8',
                -status  => 404,
            );
            $template->process(
                'bookmark.tt',
                $bookmark,
                \my $output,
            );
            return $output;
        }
    }

    # list all the bookmarks 
    my $format = $q->param('format') || 'html';
    my $tag = $q->param('tag');
    my @resources = $bookmarks->get_resources({ tag => $tag });
    my @all_tags = $bookmarks->get_tags({ selected => $tag });
    my @cotags = $bookmarks->get_cotags({ tag => $tag });

    if ($format eq 'json') {
        $self->header_props(
            -type    => 'application/json',
            -charset => 'UTF-8',
        );
        return decode_utf8(
            encode_json({
                resources => \@resources,
            })
        );
    } else {
        $self->header_props(
            -type    => 'text/html',
            -charset => 'UTF-8',
        );

        # set the base URL, adding a trailing slash if needed
        my $base_url = $q->url;
        $base_url .= '/' if $base_url =~ m{/bookmarks$};

        $template->process(
            'list.tt',
            {
                base_url     => $base_url,
                selected_tag => $tag,
                tags         => \@all_tags,
                cotags       => \@cotags,
                resources    => \@resources,
            },
            \my $output,
        );
        return $output;
    }
}

sub view {
    my $self = shift;
    my $id = $self->param('id');
    my $field = $self->param('field');

    my $bookmark = $bookmarks->get_bookmark({ id => $id });
    if ($bookmark) {
        if ($field) {
            # respond with just the requested field as plain text
            $self->header_props(
                -type    => 'text/plain',
                -charset => 'UTF-8',
            );
            my $value = $bookmark->{$field};
            return ref $value eq 'ARRAY' ? join(',', @{ $value }) : $value;
        } else {
            # display the bookmark form for this bookmark
            $bookmark->{exists} = 1;
            $bookmark->{created} = "Created " . localtime($bookmark->{ctime});
            $bookmark->{created} .= '; Updated ' . localtime($bookmark->{mtime}) unless $bookmark->{ctime} == $bookmark->{mtime};
            $self->header_props(
                -type    => 'text/html',
                -charset => 'UTF-8',
            );
            $template->process(
                'bookmark.tt',
                $bookmark,
                \my $output,
            );
            return $output;
        }
    } else {
        $self->header_props(
            -type    => 'text/html',
            -charset => 'UTF-8',
            -status  => 404,
        );
        return "Bookmark $id Not Found";
    }
}

#TODO: split this into edit and create methods
sub edit {
    my $self = shift;
    my $q = $self->query;
    #TODO: get the bookmark based on the id and edit it directly?
    #TODO: deal with changing URIs
    my $uri = $q->param('uri');
    my $title = $q->param('title');
    my @tags = split ' ', $q->param('tags');
    $bookmarks->add({
        uri   => $uri,
        title => $title,
        tags  => \@tags,
    });

=begin

    my $location = URI->new($q->url);
    $location->query_form(uri => $uri) if defined $q->url_param('uri');
    $location->fragment('updated');

=cut

    # return to the form
    $self->header_type('redirect');
    $self->header_props(
        -uri => $ENV{REQUEST_URI},
        -status => 303,
    );
}

1;
