package Bookmarks;

use Moose;
use SQL::Interp qw{:all};
use URI;
use Bookmark;

has dbh      => ( is => 'rw' );
has base_uri => ( is => 'ro', isa => 'URI' );

sub BUILD {
    my $self = shift;
    my $args = shift;

    if (!$self->dbh) {
        if ($args->{dbname}) {
            require DBI;
            $self->dbh(DBI->connect("dbi:SQLite:dbname=$$args{dbname}", "", "", { RaiseError => 1, PrintError => 0 }));
            # enable foreign key support (requires DBD::SQLite 1.26_05 or above (sqlite 3.6.19 or above))
            $self->dbh->do('pragma foreign_keys = on;');
        } else {
            #TODO: figure out how to make croak play nice with Moose to get the user-visible caller line
            die "No dbh or dbname specified in the constructor";
        }
    }
}

sub get_bookmark {
    my $self = shift;
    my $params = shift;
    my $sth;
    if ($params->{id}) {
        $sth = $self->dbh->prepare('select id,resources.uri,title,ctime,mtime from bookmarks join resources on bookmarks.uri=resources.uri where id=?');
        $sth->execute($params->{id});
    } elsif ($params->{uri}) {
        $sth = $self->dbh->prepare('select id,resources.uri,title,ctime,mtime from bookmarks join resources on bookmarks.uri=resources.uri where resources.uri=?');
        $sth->execute($params->{uri});
    } else {
        die "Must specify either id or uri";
    }
    my $bookmark = $sth->fetchrow_hashref;
    if ($bookmark) {
        my $sth_tag = $self->dbh->prepare('select tag from tags where uri = ? order by tag');
        $sth_tag->execute($bookmark->{uri});
        $bookmark->{tags} = [ map { $$_[0] } @{ $sth_tag->fetchall_arrayref } ];
        if ($self->base_uri) {
            $bookmark->{bookmark_uri} = URI->new_abs($bookmark->{id}, $self->base_uri);
        }
    }
    return $bookmark;
}

sub get_resources {
    my $self = shift;
    my $params = shift;
    my $tags = $params->{tag} || [];
    my $limit = $params->{limit};
    my $offset = $params->{offset};

    # build the query
    my @sql;

    if (!ref $tags) {
        $tags = [ $tags ];
    }
    if (@$tags) {
        my $intersect = 0;
        for my $tag (@{ $tags }) {
            push @sql, 'intersect' if $intersect;
            push @sql, 'select resources.*, bookmarks.* from resources join bookmarks on resources.uri = bookmarks.uri';
            push @sql, 'join tags on resources.uri = tags.uri where tags.tag =', \$tag;
            $intersect++;
        }
    } else {
        push @sql, 'select * from resources join bookmarks on resources.uri = bookmarks.uri';
    }
    push @sql, 'order by ctime desc';
    push @sql, ('limit', \$limit) if $limit;
    # an offset is only allowed if we have a limit clause
    push @sql, ('offset', \$offset) if $limit && $offset;

    my ($sql, @bind) = sql_interp(@sql);

    my $sth_resource = $self->dbh->prepare($sql);
    $sth_resource->execute(@bind);

    my $sth_tag = $self->dbh->prepare('select tag from tags where uri = ? order by tag');
    my @resources;
    while (my $resource = $sth_resource->fetchrow_hashref) {
        $sth_tag->execute($resource->{uri});
        $resource->{tags} = [ map { $$_[0] } @{ $sth_tag->fetchall_arrayref } ];
        if ($self->base_uri) {
            $resource->{bookmark_uri} = URI->new_abs($resource->{id}, $self->base_uri);
        }
        push @resources, $resource;
    }
    return @resources;
}

sub get_tags {
    my $self = shift;
    my $params = shift;
    my $tag = $params->{selected};
    my $sth_all_tags = $self->dbh->prepare('select tag, count(tag) as count, tag = ? as selected from tags group by tag order by tag');
    $sth_all_tags->execute($tag);
    my $all_tags = $sth_all_tags->fetchall_arrayref({});
    return @{ $all_tags };
}

sub get_cotags {
    my $self = shift;
    my $params = shift;
    my $tags = $params->{tag} || [];
    if (!ref $tags) {
        $tags = [ $tags ];
    }
    my @sql;

    push @sql, 'select tag, count(tag) as count from tags';
    if (@$tags) {
        push @sql, 'where uri in (';
        my $intersect = 0;
        for my $tag (@{ $tags }) {
            push @sql, 'intersect' if $intersect;
            push @sql, 'select uri from tags where tag = ', \$tag;
            $intersect++;
        }
        push @sql, ') and tag not in ', $tags, '';
    }
    push @sql, 'group by tag order by tag';

    my ($sql, @bind) = sql_interp(@sql);
    my $sth = $self->dbh->prepare($sql);
    $sth->execute(@bind);
    return @{ $sth->fetchall_arrayref({}) };
}

sub add {
    my $self = shift;
    my $bookmark = shift;

    my $uri = $bookmark->{uri};
    my $title = $bookmark->{title};
    #TODO: accept a ctime or mtime
    my $ctime = $bookmark->{ctime} || time;
    my $mtime = $bookmark->{mtime} || $ctime;
    my $id = $bookmark->{id};

    # create an entry for the resource
    my $sth_resource = $self->dbh->prepare('insert into resources (uri, title) values (?, ?)');
    eval {
        $sth_resource->execute($uri, $title);
    };
    if ($@) {
        if ($@ =~ /column uri is not unique/) {
            # this is not truly an error condition; the resource is already listed
            # update the title instead
            my $sth_update = $self->dbh->prepare('update resources set title = ? where uri = ?');
            $sth_update->execute($title, $uri);
        } else {
            die $@;
        }
    }

    # create the bookmark
    my ($sql_bookmark, @bind_bookmark) = sql_interp(
        'insert into bookmarks', { ($id ? (id => $id) : ()), uri => $uri, ctime => $ctime, mtime => $mtime }
    );
    my $sth_bookmark = $self->dbh->prepare($sql_bookmark);
    eval {
        $sth_bookmark->execute(@bind_bookmark);
    };
    if ($@) {
        if ($@ =~ /column uri is not unique/) {
            # this is not truly an error condition; the bookmark was already there
            # update the mtime instead
            # TODO: only update mtime if the tag list is changed?
            my $sth_update = $self->dbh->prepare('update bookmarks set mtime = ? where uri = ?');
            $sth_update->execute($mtime, $uri);
        } else {
            die $@;
        }
    }

    my %new_tags = map { $_ => 1 } @{ $bookmark->{tags} };
    my $sth_delete_tag = $self->dbh->prepare('delete from tags where uri = ? and tag = ?');
    my $sth_insert_tag = $self->dbh->prepare('insert into tags (uri, tag) values (?, ?)');
    my $sth_current_tags = $self->dbh->prepare('select tag from tags where uri = ?');
    $sth_current_tags->execute($uri);
    while (my ($tag) = $sth_current_tags->fetchrow_array) {
        if (!$new_tags{$tag}) {
            # if a current tag is not in the new tags, remove it from the database
            $sth_delete_tag->execute($uri, $tag);
        } else {
            # if a new tag is already in the database, remove it from the list of tags to add
            delete $new_tags{$tag};
        }
    }
    for my $tag (keys %new_tags) {
        $sth_insert_tag->execute($uri, $tag);
    }

=begin

    # clear all tags
    my $sth_delete_tag = $self->dbh->prepare('delete from tags where uri = ?');
    $sth_delete_tag->execute($uri);
    my $sth_tag = $self->dbh->prepare('insert into tags (uri, tag) values (?, ?)');
    for my $tag (@{ $bookmark->{tags} }) {
        #print $tag, "\n";
        # prevent duplicate (uri,tag) pairs in the database
        # TODO: should POST with a set of tags ever remove tags?
        eval {
            $sth_tag->execute($uri, $tag);
        };
        if ($@) {
            if ($@ =~ /columns uri, tag are not unique/) {
                # this is not truly an error condition; the tag was already there
            } else {
                die $@;
            }
        }
    }

=cut

    # return the newly created or updated bookmark
    return $self->get_bookmark({ uri => $uri });
}

# module returns true
1;
