package BookmarksApp;

=head1 NAME

BookmarksApp

=head1 SYNOPSIS

    use BookmarksApp;
    use Digest::MD5 qw{md5_hex};

    my $username = '...';
    my $password = '...';
    my $app = BookmarksApp->new({
        config => {
            dbname => 'bookmarks.db',

            # set these if you want non-GET requests to require authentication
            auth            => 1,
            digest_key      => 'secret',
            digest_password => md5_hex("$username:Bookmarks:$password"),
            
            # set this if the app is running behind a proxy server
            proxy_ip  => '...',
        },
    });

    # returns the coderef appropriate for use in an app.psgi,
    # or for passing the Plack::Runner, etc.
    $app->to_app;

=cut

use strict;
use warnings;

use parent qw{Plack::Component};
use Plack::Util::Accessor qw{config _app _controller};

use YAML;
use Plack::Builder;
use Plack::Request;
use Router::Resource;
use File::Basename qw{dirname};
use File::Spec::Functions qw{catfile};

use Bookmarks::Controller;

sub prepare_app {
    my $self = shift;

    my $config = $self->config;

    my $router = router {
        resource '/' => sub {
            GET {
                # check for a uri param, and if there is one present,
                # see if a bookmark for that URI already exists; if so
                # redirect to that bookmark, and if not, show the form
                # to create a new bookmark
                if (defined $self->_controller->request->param('uri')) {
                    return $self->_controller->find_or_new;
                }

                # otherwise return the sidebar
                return $self->_controller->sidebar;
            };
            POST {
                # create the bookmark and redirect to the new bookmark's edit form
                return $self->_controller->create_and_redirect;
            };
        };

        resource '/list' => sub {
            GET {
                return $self->_controller->list;
            };
        };

        resource '/feed' => sub {
            GET {
                return $self->_controller->feed;
            };
        };

        resource '/tags' => sub {
            GET {
                my ($env, $params) = @_;
                return $self->_controller->tag_tree;
            };
        };
        resource '/tags/*' => sub {
            GET {
                my ($env, $params) = @_;
                my $tag_path = (@{ $params->{splat} })[0];
                return $self->_controller->tag_tree([ split m{/}, $tag_path ]);
            };
        };

        resource '/{id}' => sub {
            GET {
                my ($env, $params) = @_;
                return $self->_controller->view($params->{id});
            };
            POST {
                my ($env, $params) = @_;
                return $self->_controller->update_and_redirect($params->{id});
            };
        };

        resource '/{id}/{field}' => sub {
            GET {
                my ($env, $params) = @_;
                return $self->_controller->view_field($params->{id}, $params->{field});
            };
        };

    };

    # if configured for auth, read in the htdigest database file
    # and store the password hashes keyed by username
    my %password_hash_for;
    if ($config->{auth}) {
        $config->{realm} ||= 'Bookmarks';
        $config->{htdigest} or die "No htdigest configured for authentication\n";

        # if authentication is enabled and no digest_key is provided, generate one
        # don't do this if it isn't needed, since this is sometimes not a fast operation
        if (!$config->{digest_key}) {
            warn "Generating digest authentication secret...\n";
            require Bytes::Random::Secure;
            $config->{digest_key} = Bytes::Random::Secure::random_bytes_base64(16, '');
        }

        open my $htdigest, '<', $config->{htdigest} or die "Can't open $$config{htdigest}\n";
        while (my $credentials = <$htdigest>) {
            chomp $credentials;
            my ($username, $realm, $password_hash) = split /:/, $credentials;
            # only add password digests for the configured realm
            if ($realm eq $config->{realm}) {
                $password_hash_for{$username} = $password_hash;
            }
        }
        close $htdigest;
    }

    $self->_app(
        builder {
            enable_if { $_[0]->{REMOTE_ADDR} eq $config->{proxy_ip} } 'ReverseProxy'
                if $config->{proxy_ip};
            enable_if { $_[0]->{REQUEST_METHOD} ne 'GET' } 'Auth::Digest', (
                realm           => $config->{realm},
                secret          => $config->{digest_key},
                password_hashed => 1,
                authenticator   => sub {
                    my ($username, $env) = @_;
                    return $password_hash_for{$username};
                },
            ) if $config->{auth};
            enable 'Static', (
                path => qr{^/assets/},
                root => catfile(dirname(__FILE__), 'htdocs'),
            );
            sub { $router->dispatch(shift); };
        }
    );
    $self->_controller(
        Bookmarks::Controller->new({
            dbname  => $self->config->{dbname},
        })
    );
}

sub call {
    my $self = shift;
    my $env = shift;

    # initialize the controller based on this request
    $self->_controller->request(Plack::Request->new($env));

    # dispatch to the app
    $self->_app->($env);
}

# module return
1;

=head1 AUTHOR

Peter Eichman <peichman@cpan.org>

=cut
